% (c) 2013-2015 Miika Aittala, Jaakko Lehtinen, Tim Weyrich, Aalto 
% University, University College London. This code is released under the 
% Creative Commons Attribution-NonCommercial-ShareAlike 4.0 International 
% license (http://creativecommons.org/licenses/by-nc-sa/4.0/).

% This procedure takes in the user-set corners of the calibration photos,
% and produces the required information about the geometric configuration.
%
% This too is messy research code, and uses partly different notation
% than the paper. It could also probably be computed much more compactly
% (and perhaps robustly) by doing clever things with points and lines at 
% inifinity, eigenvectors of homographies, etc. 
%
% Overall, we are playing with three input things: the square on the
% monitor, its reflection, plus the marker that was lying on the ground 
% plane. Horizontal direction (along the base of the monitor) is x; the
% depth into the monitor is y (the coordinate system is left handed, as a
% historical accident), and the up-direction is z.
%
% The strategy is to find the vanishing points and other relevant
% information based on the screen square and its reflection. From these we
% can ultimately build up everything we want to know. The marker square is
% used solely for determining the world scale in the y-direction;
% this is the only thing that is left ambiguous after the geometric
% construction. (It might be possible to extract the scale from the camera
% constraints too, i.e. picking the scale that gives a most plausible
% camera perspective projection.)
%
% This calibration approach has the advantage of relative simplicity for 
% the user. It does lose some resolution as one needs to take all the
% photos in the calibrated position, and that position needs to see the
% square on the screen.


function D = calibrate_1305_step_one(D)

    hold on;

    % Some helpers
    proj_line = @(x,y) cross([x(:);1], [y(:);1]);
    nlize_multi = @(x) x ./ repmat(x(end,:), [size(x,1) 1]);
    plot_point = @(x,par) plot(x(1)/x(3),x(2)/x(3),par);
    
    stdaxis = [0 0 1;
               1 0 1;
               1 1 1;
               0 1 1]';
           
           %%
           
%{
    T_marker = make_proj_transform_4(D.p_marker, stdaxis);
    l = cross(D.p_screen(1:3,1), D.p_screen(1:3,3));
    plot_line(l, D.imgwin*1, 'r');
    l2 = cross(D.p_refl(1:3,1), D.p_refl(1:3,3));
    plot_line(l2, D.imgwin*1, 'r');
    
    p = T_marker \ [0.5; 0.5; 1];
    p = p / p(3);
    plot(p(1),p(2),'ro');    
    p
  %}
    
    % The four columns of this matrix are simply 2D homogeneous coordinates
    % of a square with extents [0,1]x[0,1]
    stdaxis = ...
        [0 1 1 0;
         0 0 1 1;
         1 1 1 1];
     
    % Find one point on the hinge
    ph1 = cross( ...
        cross(D.p_screen(:,1), D.p_screen(:,3)), ...
        cross(D.p_refl(:,1), D.p_refl(:,3)) ...
        );
    
    % ... and another
    ph2 = cross( ...
        cross(D.p_screen(:,2), D.p_screen(:,4)), ...
        cross(D.p_refl(:,2), D.p_refl(:,4)) ...
        );

    % Connect the points to get the hinge line
    lh = cross(ph1,ph2);
    
    % Debug:
    plot_line(lh, D.imgwin*1, 'r');
    
    % Form a projective transformation from the [0,1]x[0,1] coordinates to
    % the perspective marker image
    T_marker = make_proj_transform_4(stdaxis, D.p_marker);
    
    % Debug: should draw a point at the center of the marker
    plot_point(T_marker * [0.5; 0.5; 1], 'ro');
    
    % Hinge line in rectified marker coordinates. Recall, points transform
    % according to A, lines transform according to A^(-T). But we want to
    % go from screen to marker, so there's a double inverse (i.e. only
    % transpose (denoted by ' in Matlab) remains)
    lhm = T_marker' * lh;
    
    % Get the slope and angle of the hinge line in marker coordinates.
    slope = -lhm(1)/lhm(2);
    ang = atan(slope);
    ANGLE_DEBUG = ang/pi*180 % print angle in degrees
    
    % Rotation matrix that rotates marker-x-directional lines to parallel 
    % with hinge, and does not touch the homogeneous stuff.
    R = [cos(ang) -sin(ang) 0; sin(ang) cos(ang) 0; 0 0 1];

    % Debug: should have zero at first index (so horizontal line)
    DEBUG = R' * lhm

    % Form the final projective transformation from marker coordinates to
    % screen coordinates, by concatenating the rotation and the earlier
    % one.
    T_marker_rot = T_marker * R;

    % Testing: draws a sequence of horizontal perspective lines, these 
    % should look parallel with the hinge
    for i = -2:0.1:2
        plot_line(inv(T_marker_rot') * [0 i 1]', D.imgwin*1, ':r');
    end
    
    % Testing: Draws the corners of the rotated marker
    for i = 1:4
        plot_point(T_marker_rot * stdaxis(:,i), 'mo');
    end
    
    %T_marker_rot
    
    return
    
    % Intersect screen and refl left edge line with hinge line
    pg1_1 = cross(cross(D.p_screen(:,1), D.p_screen(:,4)), lh);
    pg1_2 = cross(cross(D.p_refl(:,1), D.p_refl(:,4)), lh);
    pg1_1 = pg1_1 / pg1_1(3); % normalize (for next step)
    pg1_2 = pg1_2 / pg1_2(3);
    pg1 = (pg1_1 + pg1_2) / 2;  % average (for robustness)

    % Same thing for right edges
    pg2_1 = cross(cross(D.p_screen(:,2), D.p_screen(:,3)), lh);
    pg2_2 = cross(cross(D.p_refl(:,2), D.p_refl(:,3)), lh);
    pg2_1 = pg2_1 / pg2_1(3); % normalize (for next step)
    pg2_2 = pg2_2 / pg2_2(3);
    pg2 = (pg2_1 + pg2_2) / 2;  % average (for robustness)
    
    plot_point(pg1, 'ro');
    plot_point(pg2, 'ro');
    
    % Find the center of the "front face" of the extruded triangle
    pf = cross( ...
        cross(D.p_screen(:,4), D.p_refl(:,3)), ...
        cross(D.p_screen(:,3), D.p_refl(:,4)) ...
        );    
    plot_point(pf, 'rx');
    
    % Same for "back face" of the clamped extruded triangle
    pb = cross( ...
        cross(D.p_screen(:,1), D.p_refl(:,2)), ...
        cross(D.p_screen(:,2), D.p_refl(:,1)) ...
        );    
    plot_point(pb, 'rx');
    
    % Connect the two centers to get a line that passes through the center
    % of both faces, i.e. on the zx-plane along y-axis
    ly = cross(pf, pb);
    plot_line(ly, D.imgwin*1, 'b');
    
    % Find the center of the upper face of the extruded triangle
    pu = cross( ...
        cross(D.p_screen(:,4), pg2), ...
        cross(D.p_screen(:,3), pg1) ...
        );
    plot_point(pu, 'rx');
    
    % Same for lower
    pl = cross( ...
        cross(D.p_refl(:,4), pg2), ...
        cross(D.p_refl(:,3), pg1) ...
        );
    plot_point(pl, 'rx');
    
    % Connect the two. This gives a z-direction line that crosses the
    % "halfway section" (let's call it half-face) of the extruded triangle 
    % precisely at its center.
    lz = cross(pl, pu);
    
    % The intersection of this line with the earlier y-directional center
    % line gives us the centerpoint of the half-face.
    pc = cross(lz, ly);
    plot_point(pc, 'bx');
    
    % Finally, cross-connecting the hinge points to this centerpoint gives
    % a line that intersects the vertical front edges of the triangle 
    % precisely on the floor plane.
    pf1 = cross( ...
            cross(pg2, pc), ...
            cross(D.p_screen(:,4), D.p_refl(:,4)) ...
        );
    pf2 = cross( ...
            cross(pg1, pc), ...
            cross(D.p_screen(:,3), D.p_refl(:,3)) ...
        );

    plot_point(pf1, 'ro');
    plot_point(pf2, 'ro');
    
    % Phew! The polygon pg1 - pg2 - pf2 - pf1 is now actually the
    % perspective image of an axis-aligned rectangle on the floor plane.
    
    % Unfortunately it is not square and there seems to be no way to infer
    % the y-direction world scale, so we'll need to get the uniform
    % scale from the marker.
    
    
    
    return;
           
	%% Horizontal lines
    
    % Find the intersection, namely vanishing point, of the 
    % horizontal-direction (x-axis) lines, in screen coordinates.
    
    
    horiz_p = reshape([D.p_screen D.p_refl],6,[]); % note: D.p_marker removed
	
    horiz_l = [];
	
	for i = 1:size(horiz_p, 2)
        l = cross(horiz_p(1:3,i), horiz_p(4:6,i))
        %plot_line(l, D.imgwin*10, ':r');
        horiz_l = [horiz_l l];
    end
    
    function I = compute_median_isect(L)
        isects = [];
        for i = 1:size(L, 2)
            for j = i+1:size(L,2)
               %[i j] 
               isect = cross(L(:,i), L(:,j));
               isect = isect/isect(3);
               isects = [isects isect];
               plot(isect(1),isect(2),'r.')
            end
        end
 
        I = median(isects,2);
    end
    

    function I = compute_median_isect_inf(L)
        isects = [];
        for i = 1:size(L, 2)
            for j = i+1:size(L,2)
               isect = cross(L(:,i), L(:,j));
               % Normalize to unit sphere in R^3
               isect_n = isect / norm(isect);
               isects = [isects isect_n];
            end
        end
        
        I = median(isects,2);
        
        % Copy the antipodal points, so every intersection is in the list
        % twice (great circles intersect on two antipodal points)
        isects = [isects -isects]; 
        
        % Cull away one of the hemispheres
        [U,S,V] = svd(isects);
        isects = isects(:, (find(U(1,:) * isects > 0)));

        I = median(isects,2);
        I = I * sign(I(3));
    end
    
    % x-vanishing point

    p_vanishx = compute_median_isect_inf(horiz_l)
    plot(p_vanishx(1),p_vanishx(2),'ro');
    

       

    %% Refine corners with the new center
    % Overall, the calibration procedure is in need of cleanup; it is not
    % quite clear how sensitive it is to what kinds of errors. So here we
    % snap some of the input points along the rays from the vanishing
    % point. This may or may not be beneficial, but it does have the effect
    % of making some further computations "exact" (as in, precisely
    % orthogonal matrices, etc.)

    
    % Compute a projective quadrilateral with vanishing point p_vanishx as
    % center
    l_horiz1 = cross([0.1;0;1],[0.1;1;1]);
    l_horiz2 = cross([0.9;0;1],[0.9;1;1]);
    l_vanish1 = cross([0;0;1], p_vanishx);
    l_vanish2 = cross([0;1;1], p_vanishx);
    

    from = [cross(l_horiz1,l_vanish1) ...
            cross(l_horiz1,l_vanish2) ...
            cross(l_horiz2,l_vanish2) ...
            cross(l_horiz2,l_vanish1)];

    %T_vanish = make_proj_transform_4(from, stdaxis);

    % Some snapping (cryptic I know)
    yrot = [0 -1; 1 0] * (p_vanishx(1:2)/p_vanishx(3));

    T_vanish = [([yrot;0]) -p_vanishx [0;0;1]];
    T_vanish = inv(T_vanish);

    % One-off helper, uses parent context stuff
    function C = snap_vanishx4(P)
        P_v = nlize_multi(T_vanish * P);
        P_v(1,1:2) = mean(P_v(1,1:2));
        P_v(1,3:4) = mean(P_v(1,3:4));
        C = T_vanish \ P_v;
    end

    function C = snap_vanishx2(P)
        P_v = nlize_multi(T_vanish * P);
        P_v(1,:) = mean(P_v(1,:));
        C = T_vanish \ P_v;
    end

    %plot(D.p_marker(1,:), D.p_marker(2,:),'xr')
    plot(D.p_screen(1,:), D.p_screen(2,:),'xr')
    plot(D.p_refl(1,:), D.p_refl(2,:),'xr')

    %D.p_marker = nlize_multi(snap_vanishx4(D.p_marker));
     D.p_screen = nlize_multi(snap_vanishx4(D.p_screen));
    D.p_refl = nlize_multi(snap_vanishx4(D.p_refl));

    %plot(D.p_marker(1,:), D.p_marker(2,:),'xy')
    plot(D.p_screen(1,:), D.p_screen(2,:),'xy')
    plot(D.p_refl(1,:), D.p_refl(2,:),'xy')
    
	for i = 1:size(horiz_p, 2)
        %l = proj_line(horiz_p(1:2,i), horiz_p(3:4,i))
        l = cross(horiz_p(1:3,i),p_vanishx)
        plot_line(l, D.imgwin*1, 'g');
        horiz_l = [horiz_l l];
    end
    
    %% Screen-to-maker coordinate transform
    T_m = make_proj_transform_4(D.p_marker, stdaxis);

    
    %% Geometric construction

    %% Slab
    % We connect the respective points from the screen square and its
    % reflection. This gives a precisely world-vertical square slab, the
    % sides of which converge on the z-vanishing point.

    slabs = [];
    for i = 1:4
        slabline = cross(D.p_screen(:,i),D.p_refl(:,i));
        slabs = [slabs slabline];
    %    plot_line(slabline, D.imgwin,'y');
    end
    
    p_vanishz = compute_median_isect_inf(slabs);
       
    % Snap the slabs
    slabs = [];
    for i = 1:4
        slabline = cross(D.p_screen(:,i),p_vanishz);
        slabs = [slabs slabline];
        plot_line(slabline, D.imgwin,'y');
    end
    
    %% Refine screen and refl points

    % 1 and 4 are "trusted"
    D.p_screen(:,2) = cross(cross(p_vanishx,D.p_screen(:,1)), slabs(:,2));
    D.p_screen(:,3) = cross(cross(p_vanishx,D.p_screen(:,4)), slabs(:,3));
    
    D.p_refl(:,1) = cross(cross(p_vanishx,D.p_refl(:,1)), cross(p_vanishz,D.p_screen(:,1)));
    D.p_refl(:,2) = cross(cross(p_vanishx,D.p_refl(:,2)), cross(p_vanishz,D.p_screen(:,2)));
    D.p_refl(:,3) = cross(cross(p_vanishx,D.p_refl(:,3)), cross(p_vanishz,D.p_screen(:,3)));
    D.p_refl(:,4) = cross(cross(p_vanishx,D.p_refl(:,4)), cross(p_vanishz,D.p_screen(:,4)));


    %% Hinge
    % Find the screen-coordinate line of the intersection of the monitor,
    % ground and reflection planes. This can be done by intersecting the
    % lines that connect the sides of the screen and reflection squares,
    % and connecting the intersection points.
        
    
    hinge1 = cross( ... 
        cross(D.p_screen(:,1),D.p_screen(:,4)), ... 
        cross(D.p_refl(:,1),D.p_refl(:,4)));
    
    hinge2 = cross( ... % intersection of the lines between
        cross(D.p_screen(:,2),D.p_screen(:,3)), ... % screen point 2 and 3
        cross(D.p_refl(:,2),D.p_refl(:,3))); % and refl pt 2 and 3

    plot_point(hinge1,'xg');
    plot_point(hinge2,'xg');

   % ASSERT_ZERO = dot(cross(hinge1,hinge2),p_vanishx)% should be zero


    plot_point(hinge1,'og');
    plot_point(hinge2,'og');

    plot_line(cross(hinge1,D.p_screen(:,4)), D.imgwin,'g')
    plot_line(cross(hinge1,D.p_refl(:,4)), D.imgwin,':g')
    plot_line(cross(hinge2,D.p_screen(:,3)), D.imgwin,'g')
    plot_line(cross(hinge2,D.p_refl(:,3)), D.imgwin,':g')
    plot_line(cross(hinge1,hinge2), D.imgwin,'--r')

    
    %% Find the y-axis vanishing point
    
    % In 3D world coordinates, the foremost vertices of the screen and
    % reflection square can be connected to form an axis-aligned rectangle
    % that faces towards y-direction. We can compute the precise vertical
    % halfway of this rectangle in screen coordinates using a projective 
    % transformation. This gives us an x-directional line segment on the
    % floor plane, with ends precisely at same x-coordinates as the two
    % hinge points computed above. Connecting the two, the lines converge
    % at y-axis vanishing point.
    
	T_front = make_proj_transform_4([D.p_refl(:,4) D.p_refl(:,3) D.p_screen(:,3) D.p_screen(:,4)], stdaxis);

    p_fronthalf1 = T_front \ [0; 0.5; 1]
    p_fronthalf2 = T_front \ [1; 0.5; 1]
    plot_point(p_fronthalf1,'co')
    plot_point(p_fronthalf2,'co')
    hingeray1 = cross(p_fronthalf1, hinge1);
    hingeray2 = cross(p_fronthalf2, hinge2);

	plot_line(hingeray1, D.imgwin,'c');
	plot_line(hingeray2, D.imgwin,'c');
    plot_line(cross(p_vanishx, p_fronthalf1), D.imgwin,':r');
    
    p_vanishy = cross(hingeray1, hingeray2);

    % should be zero
    dot(cross(p_vanishy,p_vanishz), cross(cross(hinge1,D.p_screen(:,4)),cross(hinge2,D.p_screen(:,3))))
    
    hingeray_is1 = cross(hingeray1, slabs(:,1));    
    hingeray_is4 = cross(hingeray1, slabs(:,4));


    hingeray_is2 = cross(slabs(:,2), cross(p_vanishx, hingeray_is1));
    hingeray_is3 = cross(slabs(:,3), cross(p_vanishx, hingeray_is4));

    
    hingeray_ver2 = cross(hinge2, hingeray_is2);
    hingeray_ver3 = cross(hinge2, hingeray_is3);
    plot_line(hingeray_ver2, D.imgwin, 'c');
    plot_line(hingeray_ver3, D.imgwin, 'c');
    
    floor_proj = [hingeray_is1 hingeray_is2 hingeray_is3 hingeray_is4];
    floor_proj = nlize_multi(floor_proj);

    for i = 1:4
        %plot_point(flat(:,i),'yo');
        plot([floor_proj(1,i) floor_proj(1,mod(i,4)+1)], [floor_proj(2,i) floor_proj(2,mod(i,4)+1)],'c','LineWidth',2);
    end
    

    %% Unnormalized floor coordinate system

    % We can now form a transformation from the screen coordinates to the 
    % floor. It is metric in horizontal direction,
    % but has unknown scale in vertical.
    
    % The current "normalization" is that the projected square is of size
    % 1x1 in this (and the wall) coordinate system; so the space of
    % plausible axes is so that we set Y = ay, Z = bz, with a^2+b^2 = 1

    T_floor_unn1 = make_proj_transform_4(floor_proj, stdaxis);
    new_origin = nlize_multi(T_floor_unn1 * hinge1)
    T_floor_unn = make_proj_transform_4( ...
        [T_floor_unn1 \ (new_origin + 0), ...
         T_floor_unn1 \ (new_origin + [1;0;0]), ...
         T_floor_unn1 \ (new_origin + [1;1;0]), ...
         T_floor_unn1 \ (new_origin + [0;1;0])], ...
         stdaxis);
    
    plot_point(T_floor_unn \ [0;0;1],'og');
    plot_point(T_floor_unn \ [1;0;1],'og');
    plot_point(T_floor_unn \ [0;1;1],'og');
    
   %% Projective transformation from screen coordinates of reflection to 
   % screen coordinates of monitor
      
    T_reflect = make_proj_transform_4(...
        [hinge1 hinge2 D.p_refl(:,3) D.p_refl(:,4)], ...
        [hinge1 hinge2 D.p_screen(:,3) D.p_screen(:,4)]);
        
  
    X = -5:0.005:5;
    nX = numel(X);
    Y = sin(X*5)*2.0+0.5;
    P = [X;Y;X*0+1];
    for i = 1:nX
  %      plot_point(T_r \ P(:,i),'y.')
 %       plot_point(T_reflect * (T_r \ P(:,i)),'y.')
    end

    %% Wall coordinates
    % We can now form the zero xy-plane (in screen coordinates)
    
    wall_proj = [
        cross(cross(hinge1,p_vanishz), cross(D.p_screen(:,1),p_vanishy)) ...
        cross(cross(hinge2,p_vanishz), cross(D.p_screen(:,2),p_vanishy)) ...
        cross(cross(hinge2,p_vanishz), cross(D.p_screen(:,3),p_vanishy)) ...
        cross(cross(hinge1,p_vanishz), cross(D.p_screen(:,4),p_vanishy)) ...
        ];
    
    wall_proj = nlize_multi(wall_proj);
    for i = 1:4
        %plot_point(flat(:,i),'yo');
        plot([wall_proj(1,i) wall_proj(1,mod(i,4)+1)], [wall_proj(2,i) wall_proj(2,mod(i,4)+1)],'c','LineWidth',2);
    end

    T_wall_unn1 = make_proj_transform_4(wall_proj, stdaxis);
    new_origin = nlize_multi(T_wall_unn1 * hinge1)
    T_wall_unn = make_proj_transform_4( ...
        [T_wall_unn1 \ (new_origin + 0), ...
         T_wall_unn1 \ (new_origin + [1;0;0]), ...
         T_wall_unn1 \ (new_origin + [1;1;0]), ...
         T_wall_unn1 \ (new_origin + [0;1;0])], ...
         stdaxis);
    
    plot_point(T_wall_unn \ [0;0;1],'og');
    plot_point(T_wall_unn \ [1;0;1],'og');
    plot_point(T_wall_unn \ [0;1;1],'og');

    p_origin = T_wall_unn \ [0;0;1];
    p_x = T_wall_unn \ [1;0;1];
    p_y = T_floor_unn \ [0;1;1];
    p_z = T_wall_unn \ [0;1;1];
    

        
    %% Geometric camera estimation
    
    % By suitably connecting the vanishing points and intersecting with
    % axes, we get the camera coordinates along those axes.
        
    p_camx = cross(cross(p_origin,p_x), cross(p_vanishy, p_vanishz));
    p_camy = cross(cross(p_origin,p_y), cross(p_vanishx, p_vanishz));
    p_camz = cross(cross(p_origin,p_z), cross(p_vanishx, p_vanishy));
    
    p_camx = nlize_multi(p_camx)
    p_camy = nlize_multi(p_camy)
    p_camz = nlize_multi(p_camz)

    p_camx_f = nlize_multi(T_floor_unn * p_camx);
    p_camy_f = nlize_multi(T_floor_unn * p_camy);
    p_camz_f = nlize_multi(T_wall_unn * p_camz);
    
    % Note: not homogeneous
    p_cam_unn = [p_camx_f(1); p_camy_f(2); p_camz_f(2)]
    
    p_center_f_unn = nlize_multi(T_floor_unn * [1/2;D.aspect/2;1]);
    p_center_f_unn(3) = 0;
    
    rmat = @(a) [cos(a) -sin(a); sin(a) cos(a)];
    nlize_line = @(x) x / norm(x(1:2));

    vanish_xy_nh = nlize_multi(p_vanishx) - nlize_multi(p_vanishy);
    horizon_angle = atan(vanish_xy_nh(2) / vanish_xy_nh(1))
    vanish_xy = nlize_line(cross(p_vanishx, p_vanishy));
    horizon_dist = abs(dot(vanish_xy, [1/2; D.aspect/2; 1]))

    cam_highest = struct;
    cam_highest.err = 100000000;
    cam_highest.fit = [];
    
    % Here's a hacky 1D "optimization" that just runs through all reasonable
    % camera FOVs and picks the one that produces the most accurate
    % reprojection of the points.    
    for ang = 0.5:0.003:1.0 
        yscale = cos(ang);
        zscale = sin(ang);
        scales = [1;yscale;zscale];
        
        p_cam = p_cam_unn .* scales;
        p_center = p_center_f_unn .* scales;
        
        view_dir = p_center - p_cam;
        
        r1 = nlize_multi(T_wall_unn * wall_proj);
        r2 = nlize_multi(T_floor_unn * floor_proj);
        r = [r1(1,:);r1(2,:);r2(2,:)];
        recon = diag(scales) * r; % Reconstructed screen square
        
        % Get the camera configuration
        
        Rxy = rmat(horizon_angle);
        Rxy = [Rxy(:,1) Rxy(:,2) [0;0] ];
        Rxy = [Rxy(1,:); Rxy(2,:); [0 0 1];];

        a = [0; 0; norm(view_dir)];

        nlize2 = @(x) x / norm(x,2);

        R1 = nlize2(view_dir);
        R2 = cross(R1,[0;0;1]);
        R2 = R2 / norm(R2);
        R3 = cross(R1,R2);
        R = [R2 R3 R1];
        R = R * Rxy;
        R = R';

        a = [0; 0; norm(view_dir)];
        
        T = a - R * (p_center);

        % the point of above: [R T] maps center hit point to image center:
 	   %   [R T] * [p_center;1]

        f = horizon_dist / tan(pi/2-acos(abs(R1(3))));

        K = [1 0 1/2; 0 1 D.aspect/2; 0 0 1] * diag([-f f 1]);
        M = K * [R T];

        P = [recon (recon.*repmat([1 1 -1]',[1 4]))];% (stdaxis.*repmat([1 1 0]',[1 4])) S (S.*repmat([1 1 -1]',[1 4]))]
        
        P = [P; ones(1,size(P,2))];

        p = M * P;

        pcomp = [D.p_screen D.p_refl];
        p=nlize_multi(p);
        pcomp = nlize_multi(pcomp);
        for i = 1:size(p,2)
            plot_point(p(:,i),'y.')
%            plot_point(pcomp(:,i),'yo')
        end
        
        err = sum(sqrt(sum((p - pcomp).^2,2)));
        cam_highest.fit = [cam_highest.fit; err];
        
        if err < cam_highest.err
            cam_highest.err = err;
            cam_highest.K = K;
            cam_highest.f = f;
            cam_highest.M = M;
            cam_highest.R = R;
            cam_highest.T = T;
            cam_highest.ang = ang;
            cam_highest.recon = recon;
            
        end

    end

    D.cam = cam_highest;
    
    %% Pick an approximate scale
    
    marker_f_unn = nlize_multi(T_floor_unn * D.p_marker);
    marker_f_unn = marker_f_unn(1:2,:)
    x = marker_f_unn(1,:);
    y = marker_f_unn(2,:);
 
    % This is just a scaling that minimizes the difference of marker edge
    % lengths and corner cosines (though weighting is sort of arbitrary)
    
    % using only lines 1->2 and 2->3
%    a = sqrt(-(x(1)^2 - 2*x(1)*x(2) + 2*x(2)*x(3) - x(3)^2) / ...
 %       (y(1)^2 - 2*y(1)*y(2) + 2*y(2)*y(3) - y(3)^2));
 
    % using the opposite corner also
    a = sqrt(-(-2*x(1)*x(2) + 2*x(2)*x(3) - x(3)^2 + ...
               x(3)^2 - 2*x(3)*x(4) + 2*x(4)*x(1)) / ...
              (-2*y(1)*y(2) + 2*y(2)*y(3) - y(3)^2 + ...
               y(3)^2 - 2*y(3)*y(4) + 2*y(4)*y(1)))
    
	D.screen_angle = acos(a) / pi * 180
	% This is the proper interpretation, I guess.
    D.lcd_slope = sin(acos(a))/a;
    

    yscale = cos(acos(a));
    zscale = sin(acos(a));
    scales = [1;yscale;zscale];
    p_cam = p_cam_unn .* scales;
    D.E = p_cam;
    
    marker_f_unn(2,:) = marker_f_unn(2,:) * a;


    for i = 1:4
        this = marker_f_unn(:,i);
        prev = marker_f_unn(:,mod(i-2,4)+1);
        next = marker_f_unn(:,mod(i,4)+1);
        
        DIST = norm(this-next)
        ANGLE = acos(dot(prev-this, this-next))/pi*180        
    end
    
    %% Transformations...
    
    % From screen to floor, in metric induced by the marker, with origin at
    % hinge1, (1,0) at hinge2
    T_floor = make_proj_transform_4([ ...
        T_floor_unn \ [0;0;1] ...
        T_floor_unn \ [1;0;1] ...
        T_floor_unn \ [1;1/a;1] ...
        T_floor_unn \ [0;1/a;1]], ...
         stdaxis);

     plot_point(T_floor \ [0;1;1],'ro');
    
     
     T_lcd_unn = make_proj_transform_4([ ...
         hinge1 ...
         hinge2 ...
         D.p_screen(:,3) ...
         D.p_screen(:,4)], ...         
         stdaxis);
     
     ps = nlize_multi(T_lcd_unn * D.p_screen(:,1));
     
     % Screen to LCD, subject to same stuff as above
     T_lcd = make_proj_transform_4([ ...
         T_lcd_unn \ [0;0;1] ...
         T_lcd_unn \ [1;0;1] ...
         T_lcd_unn \ [1;1-ps(2);1] ...
         T_lcd_unn \ [0;1-ps(2);1]], ...
         stdaxis);
     
     plot_point(T_lcd \ [1;1;1],'ro');

     % Now we can compute the desired projection from floor to reflected
     % point: we have the screen->floor->lcd->screen mapping in T_reflect,
     % so just cancel the screens at edges
     
     T_reflect_floor = T_lcd * T_reflect * inv(T_floor);
     
     D.T_reflect_floor = T_reflect_floor;
     D.T_floor = T_floor;
     D.T_lcd = T_lcd;
          
     scrsize = 1/pi * 0.5; % was 0.2 in earlier; 0.15 in dec I guess 
     T_lcd_to_pi = make_proj_transform_4(D.T_lcd * D.p_screen, ...
         [[-pi*scrsize; -2*pi*scrsize; 1] ...
          [ pi*scrsize; -2*pi*scrsize; 1] ...
          [ pi*scrsize;   0; 1] ...
          [-pi*scrsize;   0; 1]]);
     
     
     
      
      
      D.T_lcd_to_pi = T_lcd_to_pi;
     
      D.T_img_to_floor = make_proj_transform_4(...
            stdaxis, ...
            D.roi_floor); 
        D.T_img_to_floor;
        
      

      T_lift_scr = make_proj_transform_4(floor_proj, D.p_screen);
      T_floor_lift_lcd = T_lcd * T_lift_scr * inv(T_floor);
      T_floor_lift_lcd = T_floor_lift_lcd / T_floor_lift_lcd(end,end);
      D.T_floor_lift_lcd = T_floor_lift_lcd;
      % Unsurprisingly the result should be something like diag(1,1/a,1)
      
      T_floor_lift_lcd_inv = inv(T_floor_lift_lcd);
      
      % Note: not homogeneous!
      Te_lcd_to_3d = [T_floor_lift_lcd_inv(1:2,1:2)]
      Te_lcd_to_3d = [Te_lcd_to_3d; D.lcd_slope * Te_lcd_to_3d(2,:)]
      D.Te_lcd_to_3d  =Te_lcd_to_3d;
      % Again unsurprisingly this should be orthonormal
      
      % This is trivial, just put zero to the last coord
      Te_floor_to_3d = [1 0; 0 1; 0 0];
      D.Te_floor_to_3d = Te_floor_to_3d;
      
      
      % Construct the similarity transformation from pi-coordinates: now the
      % world coordinates corresponding to the function coordinates x are
      % obtained as R*x+t
      t_temp = D.T_lcd_to_pi \ [0;0;1];
      i_temp = inv(D.T_lcd_to_pi);

      D.R = D.Te_lcd_to_3d * i_temp(1:2,1:2);
      D.t = D.Te_lcd_to_3d * (t_temp(1:2)/t_temp(3));
      D.s = norm(D.R(:,1));
      D.Rt = D.R / D.s;
      
      
     for i = 1:-10
         pf = [rand(2,1) * 3 - [1.5;0];1];
         p1 = nlize_multi(T_floor \ pf);
%         p2 = nlize_multi(inv(T_lcd)* T_reflect_floor * pf);
         p2 = nlize_multi(inv(T_lcd)* T_floor_lift_lcd * pf);
         p3 = nlize_multi(inv(T_lcd)* T_reflect_floor * pf);
                  
     	plot([p1(1) p2(1)],[p1(2) p2(2)],'-ro')
     	plot([p1(1) p3(1)],[p1(2) p3(2)],':bo')
        %plot(p1(1),p1(2),'ro')
         
     end

     
end



