/*
  (c) 2013-2015 Miika Aittala, Jaakko Lehtinen, Tim Weyrich, Aalto 
  University, University College London. This code is released under the 
  Creative Commons Attribution-NonCommercial-ShareAlike 4.0 International 
  license (http://creativecommons.org/licenses/by-nc-sa/4.0/).
*/


#pragma once

#include "base/Main.hpp"
#include "gpu/GLContext.hpp"
#include "base/Timer.hpp"
#include "gui/Window.hpp"
#include "gui/CommonControls.hpp"

#include <functional>

#include "Camera.hpp"


// A very very simple task sequencing mechanism: the program has a queue
// of tasks and if it is non-empty, it always calls the first one's ->draw()
// instead of the default-draw. Whether the draw() actually draws anything is up
// to implementation. draw() returns true if it has finished. If it wishes
// to do initialization, it must detect whether it is being called for the first 
// time.
class Task 
{
public:
	Task() : m_initialized(false) {};
	virtual ~Task() {};

	virtual bool draw(FW::GLContext *gl, CanonState *camera) { return true; };
	virtual bool handleEvent(const FW::Window::Event& ev) { return false; };
protected:
	bool m_initialized;
};


class LambdaTask : public Task
{
public:
	typedef std::function<bool (FW::GLContext*, CanonState*, float, bool)> Fun;
	LambdaTask(Fun fun) : Task(), m_fun(fun) {};
	virtual ~LambdaTask() {};

	virtual bool draw(FW::GLContext *gl, CanonState *camera) 
	{
		if (!m_initialized) m_timer.start();
		bool res = m_fun(gl, camera, m_timer.getElapsed(), m_initialized);
		m_initialized = true;
		return res;
	};
protected:
	//Function m_function;
	FW::Timer m_timer;
	 
	Fun m_fun;
};


class DelayTask : public Task
{
public:
	DelayTask(float delay = 0.0f) : Task(), m_delay(delay) {};
	virtual ~DelayTask() {};

	virtual bool draw(FW::GLContext *gl, CanonState *camera) 
	{ 
		if (!m_initialized)
		{
			m_timer.start();
			m_initialized = true;
		}
		glClear(GL_COLOR_BUFFER_BIT);
		if (m_timer.getElapsed() > m_delay)
			return true;
		return false;
	};

protected:
	float m_delay;
	FW::Timer m_timer;
};



class WaitMsgTask : public Task
{
public:
	WaitMsgTask(FW::CommonControls *commonCtrl, FW::String msg) : Task(), m_finished(false), m_commonCtrl(commonCtrl), m_msg(msg) {};
	virtual ~WaitMsgTask() {};

	virtual bool draw(FW::GLContext *gl, CanonState *camera) 
	{ 
		glClear(GL_COLOR_BUFFER_BIT);
		if (m_finished) 
			return true;

		m_commonCtrl->message(m_msg);

		return false;
	};

	virtual bool handleEvent(const FW::Window::Event& ev) 
	{ 
		if (ev.type == FW::Window::EventType_KeyUp)
			m_finished = true;
		return true;
	};


protected:
	FW::CommonControls *m_commonCtrl;
	FW::String m_msg;
	bool m_finished;
};


class ExposureTime
{
public:
	ExposureTime(float exposure = 1.0f, float pre = 0, float post = 0)
		: m_exposure(exposure), m_pre(pre), m_post(post) {}

	void start() 
	{ 
		m_timer.start();
	}

	float elapsedNormalized()
	{
		return (m_timer.getElapsed() - m_pre) / m_exposure;
	}
	float elapsedNormalizedCut()
	{
		float en = (m_timer.getElapsed() - m_pre) / m_exposure;
		if (en < 0 || en >= 1) en = 0;
		return en;
	}

	bool finished()
	{
		return m_timer.getElapsed() > m_pre+m_exposure+m_post;
	}

	float exposure() const { return m_exposure; }

	float m_pre, m_post, m_exposure;
	FW::Timer m_timer;
private:
};


class DisplayFrequencyTask : public Task
{
public:
	DisplayFrequencyTask(
		FW::Vec2f frequency = FW::Vec2f(1.0f,1.0f),
		float phase = 0.0f,
		ExposureTime exposure = ExposureTime(),
		FW::Vec2f window_stdi = FW::Vec2f(1.0f, 1.0f),//16.0f/10.0f),
		bool squeeze = false,
		bool threshold = true,
		float aspect = 16.0f/9.0f) : 
			Task(), 
			m_frequency(frequency), 
			m_phase(phase), 
			m_exposure(exposure),
			m_window_stdi(window_stdi), 
			m_squeeze(squeeze), 
			m_threshold(threshold),
			m_aspect(aspect)
			{};

	~DisplayFrequencyTask() {};

	virtual bool draw(FW::GLContext *gl, CanonState *camera);

protected:

	FW::Vec2f m_frequency;
	float m_phase;
	FW::Vec2f m_window_stdi;

	bool m_squeeze;
	bool m_threshold;
	float m_aspect;

	ExposureTime m_exposure;
};


class DisplayCalibrationTask : public Task
{
public:
	DisplayCalibrationTask(
		FW::Vec2f lowerLeft = FW::Vec2f(-0.5, -1.0),
		FW::Vec2f upperRight = FW::Vec2f(0.5, -0.0),
		ExposureTime exposure = ExposureTime(),
		float intensity = 0.8,
		float aspect = 16.0f/9.0f) : 
			Task(), 
			m_lowerLeft(lowerLeft),
			m_upperRight(upperRight),
			m_exposure(exposure),
			m_intensity(intensity),
			m_aspect(aspect)
			{};

	~DisplayCalibrationTask() {};

	virtual bool draw(FW::GLContext *gl, CanonState *camera);

protected:

	FW::Vec2f m_lowerLeft;
	FW::Vec2f m_upperRight;
	float m_aspect;
	float m_intensity;

	ExposureTime m_exposure;
};
class OpenShutterTask : public Task
{
public:
	OpenShutterTask() : Task() {};
	~OpenShutterTask() {};

	virtual bool draw(FW::GLContext *gl, CanonState *camera);
protected:
};


