% (c) 2013-2015 Miika Aittala, Jaakko Lehtinen, Tim Weyrich, Aalto 
% University, University College London. This code is released under the 
% Creative Commons Attribution-NonCommercial-ShareAlike 4.0 International 
% license (http://creativecommons.org/licenses/by-nc-sa/4.0/).

function [x, y] = line_to_axis(l, lims)
x = [];
y = [];
for i = 1 : size(l, 2)
    pts = line_to_axis_helper(l(:, i), lims);
    if size(pts, 2) > 0
        x = [x, pts(:, 1)];
        y = [y, pts(:, 2)];
    end
end

function x = line_to_axis_helper(l, lims)
% Returns a matrix x = [a, b] containing
% the end points of a line l clipped to 
% the region defined by lims = [xmin xmax ymin ymax].

% Some conventions:
% 
% - All the objects are in oriented projective geometry.
% 
% - A line points at the inside of the halfplane it defines.
% 
% - When we clip a part of the segment away, we force
% the intersection to be on the positive side of the 
% projective space. That is, we assume that the clipping
% region is wholly on the positive side.

% We find the points at infinity where the line intersects with the
% line at infinity. We consider the points to define a line segment
% that is then clipped to the region.
linf = [0; 0; 1];
a = cross(l, linf);
b = -a;

xmin = lims(1);
xmax = lims(2);
ymin = lims(3);
ymax = lims(4);

% Construct the bounding half spaces
x1 = [xmin; ymin; 1];
x2 = [xmax; ymin; 1];
x3 = [xmax; ymax; 1];
x4 = [xmin; ymax; 1];

s1 = cross(x1, x2);
s2 = cross(x2, x3);
s3 = cross(x3, x4);
s4 = cross(x4, x1);
s = [s1, s2, s3, s4];

for i = 1 : size(s, 2)
    si = s(:, i);
    
    if (si' * a) > 0
        % a is inside
        if (si' * b) > 0
            % both a and b are inside => nothing to do
        else
            % a is inside, b is outside
            b = cross(l, si);
            if (b(3) < 0)
                b = -b;
            end
        end
    else
        % a is outside
        if (si' * b) > 0
            % a is outside, b is inside
            a = cross(l, si);
            if (a(3) < 0)
                a = -a;
            end
        else
            % a is outside, b is outside
            x = [];
            return;
        end
    end
end

a = a(1:2) / a(3);
b = b(1:2) / b(3);
x = [a, b];
