/*
  (c) 2013-2015 Miika Aittala, Jaakko Lehtinen, Tim Weyrich, Aalto 
  University, University College London. This code is released under the 
  Creative Commons Attribution-NonCommercial-ShareAlike 4.0 International 
  license (http://creativecommons.org/licenses/by-nc-sa/4.0/).
*/


#include <stdio.h>

#include "Camera.hpp"

// http://songuke.blogspot.fi/2012/05/short-note-about-using-canon-edsdk.html


static EdsError EDSCALLBACK  handleStateEventRedirect (
						EdsUInt32			inEvent,
						EdsUInt32			inParam, 
						EdsVoid *			inContext
						)
{
	//printf("State event\n");
	if (inContext)
		return ((CanonState*)inContext)->handleStateEvent(inEvent, inParam);
	return EDS_ERR_OK;
}

static EdsError EDSCALLBACK handleObjectEventRedirect( EdsObjectEvent inEvent,
									   EdsBaseRef inObject,
									   EdsVoid * inContext)
{
//	printf("Object event\n");
	if (inContext)
		return ((CanonState*)inContext)->handleObjectEvent(inEvent, inObject);

	if(inObject)
	{
		EdsRelease(inObject);
	}

	return EDS_ERR_OK;
}

static EdsError EDSCALLBACK  handlePropertyEventRedirect(
						EdsUInt32			inEvent,
						EdsUInt32			inPropertyID,
						EdsUInt32			inParam, 
						EdsVoid *			inContext				
						)
{
	if (inContext)
		return ((CanonState*)inContext)->handlePropertyEvent(inEvent, inPropertyID, inParam);
	//std::cout << "Property event " << inEvent << ", propid " << inPropertyID << ", param " << inParam << "\n";

	return EDS_ERR_OK;
}



EdsError CanonState::getFirstCamera(EdsCameraRef *camera)
{
	EdsError err = EDS_ERR_OK;
	EdsCameraListRef cameraList = NULL;
	EdsUInt32 count = 0;
	// Get camera list
	err = EdsGetCameraList(&cameraList);
	// Get number of cameras
	if(err == EDS_ERR_OK)
	{
		err = EdsGetChildCount(cameraList, &count);
		if(count == 0)
		{
			err = EDS_ERR_DEVICE_NOT_FOUND;
		}
	}
	// Get first camera retrieved
	if(err == EDS_ERR_OK)
	{
		err = EdsGetChildAtIndex(cameraList , 0 , camera);
	}
	// Release camera list
	if(cameraList != NULL)
	{
		EdsRelease(cameraList);
		cameraList = NULL;
	}
	return err;
}



EdsError CanonState::setup()
{
	EdsError err; 

	// Initialize SDK
	err = EdsInitializeSDK();

	if(err == EDS_ERR_OK)
	{
		m_isSDKLoaded = true;
	} else
	{
		printf("Error initializing SDK\n");
		return err;
	}

	// Get first camera
	if(err == EDS_ERR_OK)
	{
		err = getFirstCamera (&m_camera);
	} else
	{
		printf("Error getting camera\n");
		return err;
	}	

	printf("Camera ok.\n");

#ifdef USE_CAMERA_EVENTS
	// Set event handler
	if(err == EDS_ERR_OK)
	{
		err = EdsSetObjectEventHandler(m_camera, kEdsObjectEvent_All,
			handleObjectEventRedirect, this);
	}
	
	// Set event handler
	if(err == EDS_ERR_OK)
	{
		err = EdsSetPropertyEventHandler(m_camera, kEdsPropertyEvent_All,
			handlePropertyEventRedirect, this);
	}
	

	// Set event handler
	if(err == EDS_ERR_OK)
	{
		err = EdsSetCameraStateEventHandler(m_camera, kEdsStateEvent_All,
			handleStateEventRedirect, this);
	}
#endif

	// operations
	if(err == EDS_ERR_OK)
	{
		err = EdsOpenSession(m_camera);
	}
	else
	{
		printf("Error opening session\n");
		//exit(1);
	}	
	
	return err;
}

EdsError CanonState::transferImage(EdsBaseRef object)
{
	printf("Transfering image\n");

	EdsStreamRef stream = NULL;
	EdsDirectoryItemRef PictureFromCamBuffer = object;
	EdsDirectoryItemInfo PictureFromCamBufferInfo;

	EdsGetDirectoryItemInfo(PictureFromCamBuffer, &PictureFromCamBufferInfo);

	//Create Outputfile(s)
	EdsCreateFileStream(PictureFromCamBufferInfo.szFileName, kEdsFileCreateDisposition_CreateAlways, kEdsAccess_ReadWrite, &stream);

	EdsDownload( PictureFromCamBuffer, PictureFromCamBufferInfo.size, stream);
	EdsDownloadComplete(PictureFromCamBuffer);

	// Release stream
	if (stream != NULL) {
		EdsRelease(stream);
		stream = NULL;
	}
	return EDS_ERR_OK;
}

void CanonState::close()
{

	EdsCloseSession(m_camera);

	if(m_camera != NULL)
	{
		EdsRelease(m_camera);
	}

	if(m_isSDKLoaded)
	{
		EdsTerminateSDK();
	}
}
