/*
  (c) 2013-2015 Miika Aittala, Jaakko Lehtinen, Tim Weyrich, Aalto 
  University, University College London. This code is released under the 
  Creative Commons Attribution-NonCommercial-ShareAlike 4.0 International 
  license (http://creativecommons.org/licenses/by-nc-sa/4.0/).
*/

#define _CRT_SECURE_NO_WARNINGS

#include "App.hpp"
#include "base/Main.hpp"
#include "gpu/GLContext.hpp"
#include "3d/Mesh.hpp"
#include "io/File.hpp"
#include "io/StateDump.hpp"
#include "base/Random.hpp"


#include <stdio.h>
#include <conio.h>
#include <map>
#include <queue>

using namespace FW;

//------------------------------------------------------------------------

App::App(void)
:   m_commonCtrl		(CommonControls::Feature_Default & ~CommonControls::Feature_RepaintOnF5),
    m_cameraCtrl		(&m_commonCtrl, CameraControls::Feature_Default | CameraControls::Feature_StereoControls),
    m_action			(Action_None)
{
	static bool camera_setup = false;
	if (!camera_setup) {
		m_canon.setup();
		camera_setup = true;
	}

	m_threshold = true;
	
	m_commonCtrl.showFPS(false);
    m_commonCtrl.addStateObject(this);
    m_cameraCtrl.setKeepAligned(true);

	//m_commonCtrl.addToggle(&m_threshold, FW_KEY_NONE, "Use thresholding" );
	//m_commonCtrl.addSeparator();

	m_commonCtrl.addButton((S32*)&m_action, Action_ReconnectCamera,                FW_KEY_C,       "Reconnect camera");
	m_commonCtrl.addSeparator();

    m_commonCtrl.addButton((S32*)&m_action, Action_TestButton,                FW_KEY_T,       "(debug test)");
	m_commonCtrl.addButton((S32*)&m_action, Action_Shoot,                FW_KEY_S,       "Shoot camera");
	
	m_commonCtrl.addSeparator();
    m_commonCtrl.addButton((S32*)&m_action, Action_ShootExposure,                FW_KEY_NONE,       "Shoot exposure photo");
    m_commonCtrl.addButton((S32*)&m_action, Action_ShootCalibration,                FW_KEY_NONE,       "Shoot calibration photo");
    m_commonCtrl.addButton((S32*)&m_action, Action_ShootMarker,                FW_KEY_NONE,       "Shoot marker photo");
    m_commonCtrl.addButton((S32*)&m_action, Action_ShootStandard,                FW_KEY_NONE,       "Shoot standard frequency program");

	m_commonCtrl.beginSliderStack();
    m_commonCtrl.addSlider(&m_exp, 0, 10.0f, false, FW_KEY_NONE, FW_KEY_NONE, "Exposure time= %f");
    m_commonCtrl.addSlider(&m_expPre, 0, 2.0f, false, FW_KEY_NONE, FW_KEY_NONE, "Pre-exposure buffer time= %f");
    m_commonCtrl.addSlider(&m_expPost, 0, 4.0f, false, FW_KEY_NONE, FW_KEY_NONE, "Post-exposure buffer time= %f");

    m_commonCtrl.endSliderStack();

	m_commonCtrl.beginSliderStack();
    m_commonCtrl.addSlider(&m_windowSize, 0, 4.0f, false, FW_KEY_NONE, FW_KEY_NONE, "Window size= %f");

    m_commonCtrl.endSliderStack();

    m_window.setTitle("Capture");
    m_window.addListener(this);
    m_window.addListener(&m_commonCtrl);
	m_window.toggleFullScreen();
	// all state files fill be prefixed by this; enables loading/saving between debug and release builds
	m_commonCtrl.setStateFilePrefix( "state_capture_" );

//    m_commonCtrl.loadState(m_commonCtrl.getStateFileName(1));

	setupShaders( m_window.getGL() );

	glClearColor(0.0f, 0.0f, 0.0f, 1.0f);

	m_exp = 3.2f;
	m_expPre = 0.5f;
	m_expPost = 2.0f;
	m_threshold = true;
	m_windowSize = 2.0f;
	m_calibRectLL = Vec2f(-0.5f,-1.0f);
	m_calibRectUR = Vec2f(0.5f, 0.0f);
	m_windowMean = Vec2f(0,0);
	m_windowStdi = Vec2f(1.0f, getAspect());

}

//------------------------------------------------------------------------

App::~App(void)
{
}

//------------------------------------------------------------------------

bool App::handleEvent(const Window::Event& ev)
{

	m_threshold = false;
	if (ev.type == Window::EventType_Close)
    {
        m_window.showModalMessage("Exiting...");
        delete this;
        return true;
    }

    Action action = m_action;
    m_action = Action_None;
    String name;
    Mat4f mat;

	// Standard program parameters (can't be declared inside switch())
	const static float freqs[8] = {1,     2,     3     ,4,     6,    10,    20    ,40};
	const static float phases[6] = {0, 1.57079633f, 3.14159265f, 4.71238898f};
	const static int xseq[4] = {1, 1, 0, -1};
	const static int yseq[4] = {0, 1, 1, 1};

	m_windowStdi = Vec2f(1.0f / m_windowSize, getAspect() / m_windowSize);


    switch (action)
    {
    case Action_None:
        break;
    case Action_ReconnectCamera:
		m_canon.close();
		m_canon.setup();
		break;

    case Action_Shoot:
		m_taskqueue.push_back(new OpenShutterTask());
        break;
		
    case Action_TestButton:
		/*
//		m_taskqueue.push_back(new Task());
		m_taskqueue.push_back(new OpenShutterTask());
		m_taskqueue.push_back(new DisplayFrequencyTask(FW::Vec2f(0,0)));

		m_taskqueue.push_back(new DelayTask(0.5f));
		m_taskqueue.push_back(new DisplayCalibrationTask(m_calibRectLL, m_calibRectUR, 0.8f, getAspect()));

		m_taskqueue.push_back(new WaitMsgTask(&m_commonCtrl, "Press any key to continue."));

		m_taskqueue.push_back(new LambdaTask(
			[] (FW::GLContext* gl, CanonState* cam, float time, bool inited) -> bool
			{
				glClearColor(0.0f, 0.0f, 0.0f, 1.0f);
				glClear(GL_COLOR_BUFFER_BIT);
				if (time > 2.0f) return true;
				return false;
			}));
		
		m_taskqueue.push_back(new OpenShutterTask());
		m_taskqueue.push_back(new DisplayFrequencyTask(FW::Vec2f(10,10)));
		m_taskqueue.push_back(new OpenShutterTask());
		m_taskqueue.push_back(new DisplayFrequencyTask(FW::Vec2f(0,10)));
		*/
		for (int o = 0; o < 16; o++)
		{
					m_taskqueue.push_back(new DelayTask(0.05));
					m_taskqueue.push_back(new OpenShutterTask());
					m_taskqueue.push_back(new DelayTask(m_expPre));
					m_taskqueue.push_back(new DisplayFrequencyTask(FW::Vec2f(20,20), 0, ExposureTime(m_exp), m_windowStdi, true, true, getAspect()));
					m_taskqueue.push_back(new DelayTask(m_expPost));
		}
		
		for (int o = 0; o < 16; o++)
		{
					m_taskqueue.push_back(new DelayTask(0.05));
					m_taskqueue.push_back(new OpenShutterTask());
					m_taskqueue.push_back(new DelayTask(m_expPre));
					m_taskqueue.push_back(new DisplayFrequencyTask(FW::Vec2f(20,20), 0, ExposureTime(m_exp), m_windowStdi, false, true, getAspect()));
					m_taskqueue.push_back(new DelayTask(m_expPost));
		}
		
        break;

    case Action_ShootExposure:
		m_commonCtrl.showControls(false);
		m_taskqueue.push_back(new DelayTask(0.05));
		m_taskqueue.push_back(new OpenShutterTask());
		m_taskqueue.push_back(new DelayTask(m_expPre));
		m_taskqueue.push_back(new DisplayFrequencyTask(FW::Vec2f(0,0), 3.1415926535f/2.0f, ExposureTime(m_exp), m_windowStdi, false, m_threshold, getAspect()));
		m_taskqueue.push_back(new DelayTask(m_expPost));
		break;

    case Action_ShootCalibration:
		m_commonCtrl.showControls(false);
		m_taskqueue.push_back(new DelayTask(0.05));
		m_taskqueue.push_back(new OpenShutterTask());
		m_taskqueue.push_back(new DelayTask(m_expPre));
		m_taskqueue.push_back(new DisplayCalibrationTask(m_calibRectLL, m_calibRectUR, ExposureTime(m_exp), 0.3f, getAspect()));
		m_taskqueue.push_back(new DelayTask(m_expPost));
		break;


    case Action_ShootMarker:
		m_commonCtrl.showControls(false);
		m_taskqueue.push_back(new DelayTask(0.05));
		m_taskqueue.push_back(new OpenShutterTask());
		m_taskqueue.push_back(new DelayTask(m_expPre));
		m_taskqueue.push_back(new DisplayFrequencyTask(FW::Vec2f(0,0), 3.1415926535f/2.0f, ExposureTime(m_exp), m_windowStdi, false, m_threshold, getAspect()));
		m_taskqueue.push_back(new DelayTask(m_expPost));
		break;


    case Action_ShootStandard:
		m_commonCtrl.showControls(false);

		// Black frame
		m_taskqueue.push_back(new DelayTask(0.05));
		m_taskqueue.push_back(new OpenShutterTask());
		m_taskqueue.push_back(new DelayTask(m_expPre));
		m_taskqueue.push_back(new DisplayFrequencyTask(FW::Vec2f(0,0), 0, ExposureTime(m_exp), m_windowStdi, false, m_threshold, getAspect()));
		m_taskqueue.push_back(new DelayTask(m_expPost));

		// White frame (DC)
		m_taskqueue.push_back(new DelayTask(0.05));
		m_taskqueue.push_back(new OpenShutterTask());
		m_taskqueue.push_back(new DelayTask(m_expPre));
		m_taskqueue.push_back(new DisplayFrequencyTask(FW::Vec2f(0,0), 3.1415926535f/2, ExposureTime(m_exp), m_windowStdi, false, m_threshold, getAspect()));
		m_taskqueue.push_back(new DelayTask(m_expPost));

		for (int o = 0; o < 4; o++)
		{
			for (int f = 0; f < 8; f++)
			{
				for (int p = 0; p < 4; p++)
				{
					float fx = freqs[f] * xseq[o];
					float fy = freqs[f] * yseq[o];
					float phase = phases[p];

					printf("frequency (%f, %f), phase %f\n", fx, fy, phase);
					m_taskqueue.push_back(new DelayTask(0.05));
					m_taskqueue.push_back(new OpenShutterTask());
					m_taskqueue.push_back(new DelayTask(m_expPre));
					m_taskqueue.push_back(new DisplayFrequencyTask(FW::Vec2f(fx,fy), phase, ExposureTime(m_exp), m_windowStdi, false, m_threshold, getAspect()));
					m_taskqueue.push_back(new DelayTask(m_expPost));
				}
			}
		}
		m_taskqueue.push_back(new DelayTask(3.0f));
		break;

	default:
        FW_ASSERT(false);
        break;
    }

	/*
	if (action == Action_None && ev.type == Window::EventType_KeyUp)
	{
		
	}
	*/

	if (m_taskqueue.size() > 0)
	{
		m_taskqueue.front()->handleEvent(ev);
	}

	m_window.setVisible(true);


    if (ev.type == Window::EventType_Paint)
	{
		if (m_taskqueue.size() > 0)
		{
			bool finished = m_taskqueue.front()->draw(m_window.getGL(), &m_canon);
			if (finished) {
				delete m_taskqueue.front();
				m_taskqueue.pop_front();
				//printf("Task finished.\n");
			}
		}
		else
		{
			m_commonCtrl.showControls(true);
	        renderFrame(m_window.getGL());
			

		}
	}
    m_window.repaint();

	/*
    MSG msg;	
    // init SDK, send take picture command
	
    // event loop to dispatch message in win32
    if(GetMessage(&msg, NULL, NULL, NULL))
    {        
        TranslateMessage(&msg); 
        DispatchMessage(&msg); 
    }
	*/
//	setDiscardEvents(false);


    return false;
}

//------------------------------------------------------------------------

void App::readState(StateDump& d)
{
    String meshFileName;

    d.pushOwner("App");
    d.popOwner();

}

//------------------------------------------------------------------------

void App::writeState(StateDump& d) const
{
    d.pushOwner("App");
    d.popOwner();

}

//------------------------------------------------------------------------

void App::waitKey(void)
{
    printf("Press any key to continue . . . ");
    _getch();
    printf("\n\n");
}

//------------------------------------------------------------------------

void App::renderFrame(GLContext* gl)
{
	// Setup transformations.

	Mat4f worldToCamera = m_cameraCtrl.getWorldToCamera();
	Mat4f projection = gl->xformFitToView(Vec2f(-1.0f, -1.0f), Vec2f(2.0f, 2.0f)) * m_cameraCtrl.getCameraToClip();

	// Initialize GL state.

	glClearColor(0.0f, 0.0f, 0.0f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glDisable(GL_DEPTH_TEST);

	glDisable(GL_CULL_FACE);

	// setup up tone mapping
	GLContext::Program* prog = gl->getProgram("MeshBase::draw_generic");
	prog->use();

	// Render.

	//renderScene(gl, worldToCamera, projection);
	renderQuad(gl);

	// Display status line.
/*
	m_commonCtrl.message(sprintf("asdf"
		),
		"meshStats");
		*/
}

//------------------------------------------------------------------------

void App::renderScene(GLContext* gl, const Mat4f& worldToCamera, const Mat4f& projection)
{
    // Draw mesh.
//    if (m_mesh)
  //      m_mesh->draw(gl, worldToCamera, projection);
}

void App::renderQuad(GLContext* gl)
{


	float M_PI = 3.1415926;
	float gw = 1920;
	float gh = 1280;
	float gdiff = (gw-gh)/2;
	float aspect = gh/gw;
#if 0
	glViewport(0,0,gw,gh);
	//Go to projection mode and load the identity
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	//Orthographic projection, stretched to fit the screen dimensions 
	gluOrtho2D(0,gw,0,gh);
	//Go back to modelview and load the identity
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	glBegin(GL_POLYGON);
			glTexCoord2i(-M_PI, -M_PI); glVertex2i(0, -gdiff);
			glTexCoord2i(M_PI, -M_PI); glVertex2i(gw, -gdiff);
			glTexCoord2i(M_PI,M_PI); glVertex2i(gw, gh+gdiff);
			glTexCoord2i(-M_PI, M_PI); glVertex2i(0, gh+gdiff);
	glEnd();
#endif
	//

	F32 posAttrib[] =
    {
		-1, -1, 0, 1,
		1, -1, 0, 1,
		-1, 1, 0, 1,
		1, 1, 0, 1
    };

    F32 texAttrib[] =
    {
		-M_PI, -M_PI/getAspect(),
		M_PI, -M_PI/getAspect(),
		-M_PI, M_PI/getAspect(),
		M_PI, M_PI/getAspect()
    };

    // Create program.

    static const char* progId = "App::drawFullScreenQuad";
    GLContext::Program* prog = gl->getProgram(progId);
    if (!prog)
    {
        prog = new GLContext::Program(
            FW_GL_SHADER_SOURCE(
                attribute vec4 posAttrib;
                attribute vec2 texAttrib;
                varying vec2 W;
                void main()
                {
                    gl_Position = posAttrib;
                    W = texAttrib;
                }
            ),
            FW_GL_SHADER_SOURCE(
                uniform sampler2D texSampler;
                varying vec2 W;

				uniform float threshold;
				uniform vec2 lowerLeft;
				uniform vec2 upperRight;
				uniform float time;
				uniform vec2 window_covi;

                void main()
                {
//                    gl_FragColor = vec4(vec3(sin(W.s)*0.5+0.5>mod(threshold,1.0)), 1);
//                    gl_FragColor = vec4(vec3((sin(5*W.s)*0.5+0.5-mod(threshold,1.0))*120), 1);
					vec3 val = 0;

					if (W.x > lowerLeft.x && W.x < upperRight.x && W.y > lowerLeft.y && W.y < upperRight.y)
						val += vec3(0.8,0,0);


					float win = dot(window_covi * W, window_covi * W);

					//if (W.x > -1 && W.x < 1 && W.y > -1 && W.y < 1)
					if (win < 1.0)
					{
						val += vec3(0,0.5,0) * vec3(sin(2*time+W.x*5)>0);
						val += vec3(0,0.5,0) * vec3(sin(2*time+W.y*5)>0);
						val += vec3(0,0,0.5) * vec3(sin(-2*time+W.x*20)>0);
						val += vec3(0,0,0.5) * vec3(sin(-2*time+W.y*20)>0);
					}

					if (dot(W,W) < 0.01) val += vec3(1.0,1.0,1.0);

					gl_FragColor = vec4(val, 1);
                }
            ));
        gl->setProgram(progId, prog);
    }

    prog->use();

    gl->setAttrib(prog->getAttribLoc("posAttrib"), 4, GL_FLOAT, 0, posAttrib);
    gl->setAttrib(prog->getAttribLoc("texAttrib"), 2, GL_FLOAT, 0, texAttrib);

	float elapsedTime = m_timer.getElapsed()*0.33;
	gl->setUniform(prog->getUniformLoc("threshold"), elapsedTime);

	gl->setUniform(prog->getUniformLoc("lowerLeft"), m_calibRectLL);
	gl->setUniform(prog->getUniformLoc("upperRight"), m_calibRectUR);

	gl->setUniform(prog->getUniformLoc("window_covi"), m_windowStdi);
	gl->setUniform(prog->getUniformLoc("time"), (float)(GetTickCount()) / 1000.0f);

	glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
}

//------------------------------------------------------------------------


void App::setupShaders( GLContext* gl )
{
    GLContext::Program* prog = new GLContext::Program(
        "#version 120\n"
        FW_GL_SHADER_SOURCE(
            uniform mat4 posToClip;
            uniform mat4 posToCamera;
            uniform mat3 normalToCamera;
            attribute vec3 positionAttrib;
            attribute vec3 normalAttrib;
            attribute vec4 vcolorAttrib; // Workaround. "colorAttrib" appears to confuse certain ATI drivers.
            attribute vec2 texCoordAttrib;
            centroid varying vec3 positionVarying;
            centroid varying vec3 normalVarying;
            centroid varying vec4 colorVarying;
            varying vec2 texCoordVarying;

            void main()
            {
                vec4 pos = vec4(positionAttrib, 1.0);
                gl_Position = posToClip * pos;
                positionVarying = (posToCamera * pos).xyz;
                normalVarying = normalToCamera * normalAttrib;
                colorVarying = vcolorAttrib;
                texCoordVarying = texCoordAttrib;
            }
        ),
        "#version 120\n"
        FW_GL_SHADER_SOURCE(
            uniform bool hasNormals;
            uniform bool hasDiffuseTexture;
            uniform bool hasAlphaTexture;
            uniform vec4 diffuseUniform;
            uniform vec3 specularUniform;
            uniform float glossiness;
			uniform float reinhardLWhite;
			uniform float tonemapBoost;
            uniform sampler2D diffuseSampler;
            uniform sampler2D alphaSampler;
            centroid varying vec3 positionVarying;
            centroid varying vec3 normalVarying;
            centroid varying vec4 colorVarying;
            varying vec2 texCoordVarying;

            void main()
            {
                vec4 diffuseColor = diffuseUniform;
                vec3 specularColor = specularUniform;

                if (hasDiffuseTexture)
                    diffuseColor.rgb = texture2D(diffuseSampler, texCoordVarying).rgb;

				diffuseColor *= colorVarying;

                if (hasAlphaTexture)
                    diffuseColor.a = texture2D(alphaSampler, texCoordVarying).g;

                if (diffuseColor.a <= 0.5)
                    discard;

				diffuseColor *= tonemapBoost;
				float I = diffuseColor.r + diffuseColor.g + diffuseColor.b;
				I = I*(1.0f/3.0f);
				diffuseColor *= ( 1.0f + I/(reinhardLWhite*reinhardLWhite) ) / (1 + I);

                gl_FragColor = diffuseColor;
            }
        ));

	gl->setProgram( "MeshBase::draw_generic", prog );
}



//------------------------------------------------------------------------

void FW::init(void)
{
    new App;
}

//------------------------------------------------------------------------

bool App::fileExists( const String& fn )
{
	FILE* pF = fopen( fn.getPtr(), "rb" );
	if( pF != 0 )
	{
		fclose( pF );
		return true;
	}
	else
	{
		return false;
	}
}

//------------------------------------------------------------------------
