function [k, cshift, f, fshift, ferror, error] = DispersionErrorShift(fs,length_ax,c,n_ax,plotfig);
% Function to calculate the % error (in Hz) at a specific frequency for 
% a rectilinear 3D FDTD grid based on normalised dispersion error factor
%
% n_ax      :   axial mode number of interest
% length_ax :   Room dimenstion related to mode (m)
% fs        :   temporal sampling rate of FDTD grid (Hz)
% c         :   speed of sound used (m/s)
%
% k         :   normalized dispersion error factor
% cshift    :   new speed of sound (m/s)
% f         :   original mode frequency (Hz)
% fshift    :   shifted version (Hz)
% ferror:   :   amount of shift (Hz)
% error     :   Percentage shift

%% Initialise parameters
pi = 3.14159;
pi2 = 2*pi;

%%Calculate frequency of axial mode
f =(c/2)*(n_ax/length_ax);

normxlim = f/fs;
incr = normxlim./100;

%% Set up 2D dispersion error plot axes
x = -normxlim:incr:normxlim;
y = -normxlim:incr:normxlim;
[X,Y] = meshgrid(x,y);

%% Set up 3D dispersion error plot axes
k = 8;
n = 2^k-1;
step = 0.5;
theta = (pi)*(-n:step:n)/n;
phi = (pi/2)*(-n:step:n)'/n;

X3D = (normxlim)*cos(phi)*cos(theta);
Y3D = (normxlim)*cos(phi)*sin(theta);
Z3D = (normxlim)*sin(phi)*ones(size(theta));

%% Calculate independent variable - 2D plane first 
% for z -> 0 (z = 0 undefined)

Z1 = 0.00001;
C1 = (sqrt(3.0)./(pi2*sqrt(X.^2 + Y.^2 + Z1.^2))).*acos((cos(pi2*X) + cos(pi2*Y) + cos(pi2*Z1))./3.0);

% Give max/min values of disperion error factor.
max(max(C1));
min(min(C1));

%% Calculate independent variable - 3D surface 
C = (sqrt(3.0)./(pi2*sqrt(X3D.^2 + Y3D.^2 + Z3D.^2))).*acos((cos(pi2*X3D) + cos(pi2*Y3D) + cos(pi2*Z3D))./3.0);

% Give max/min values of disperion error factor.
max(max(C));
k = min(min(C)); % This is the value neded

cshift = c*k; %Calculate change in c
fshift =(cshift/2)*(n_ax/length_ax); % Calculate new resonant mode frequency

ferror = abs(f-fshift); % Frequency error
error = 100*ferror/f; % Percentage error

%% Plotting commands
if nargin == 5
    % 2D Plane
    figure(1);
    clf;

    %background of the window - white
    set(gcf,'Color',[1,1,1])
    % 30 successive contour steps of 1 percent
    % v = 0.7:0.01:1;
    plot(141);
    surfc(X, Y, C1);
    %contourf(X,Y,C1,v);
    % caxis auto % for automatically scaled map. Below sets it to a specific
    % scale
    caxis ([0.9 1]);

    set(gca,'XLim',[-normxlim normxlim],'YLim',[-normxlim normxlim]);
    %set(gca,'XTick',-normxlim:normxlim:normxlim,'XTickLabel',{'-0.5','0','0.5'});
    %set(gca,'YTick',-normxlim:normxlim:normxlim,'YTickLabel',{'-0.5','0','0.5'});
    %set(gca, 'FontSize',6);
    %%set(gca, 'Fontname' some font allowing greek characters?
    axis square;

    % 3D Surface
    figure(2);

    % This command creates the plot
    SU = surf(X3D,Y3D,Z3D,C,'EdgeColor','none');
    %colormap('gray');
    % caxis auto % for automatically scaled map. Below sets it to a specific
    % scale
    caxis ([0.9 1]);

    axis equal; axis tight;
    set(gca,'XLim',[-normxlim normxlim],'YLim',[-normxlim normxlim],'ZLim',[-normxlim normxlim]);
    %set(gca,'XTick',-normxlim:normxlim:normxlim,'XTickLabel',{'-0.5','0','0.5'});
    %set(gca,'YTick',-normxlim:normxlim:normxlim,'YTickLabel',{'-0.5','0','0.5'});
    %set(gca,'ZTick',-normxlim:normxlim:normxlim,'ZTickLabel',{'-0.5','0','0.5'});

    %background of the window - white
    set(gcf,'Color',[1,1,1])
    view(30,35);
end
%%

