function IEEECalibrationProcessData( fs,d,fc,mode,C )
% This function should be called from IEEEHybridCalibrationMain.m
%
% Alex Southern
% Virtual Acoustics Team
% Department of Media Technology,
% Aalto University,
% Otaniemi,
% Espoo,
% Finland
% 47001

display(' ');
display('Creating max error result...'); 
getMaxErrorCutOffFreqFigure( fs,d,fc );

% Obtain calibration parameter from syntheised (measured) receiver data
display('Producing Table 1(a)...'); 
Ta = getMeasuredData( fs,d,0.01 );

if mode == 0
    % Fit a straight line to measured calibration parameter data
    for j = 1:length(d)
        C = polyfit( fs, Ta(1,:,j),1);    
        display(['Calibration line based on ' num2str(d(j)) 'm measurements: Wfs x ' num2str(C(1)) ' + ' num2str(C(2))]);
    end
end

% Use coefficients to calibrate the dataset with a predicted calibration
% parameter.
display('Producing Table 1(b)...'); 
Tb = getPredictedData( fs,d,0.01,C );

% Plot measured and predicted calibration parameter data in Table 1 for
% convenience.
for j = 1:length(d)
    figure;
    plot(fs,Ta(1,:,j),'.-');
    hold on;
    plot(fs,Tb(1,:,j),'.-r');
    xlabel('Sampling Frequency (Hz)');
    ylabel('Calibration Parameter');
    title(['Measured (blue) and Predicted (red) Calibration Parameters based on ' num2str(d(j)) 'm measurements']); 
end

% Produce Table for Article
for j = 1:length(d);
    T = Ta(:,:,j);
    T(4:6,:) = Tb(:,:,j);
    f = figure('Position',[200 200 700 200]);

    for i = 1:length(fs)
        str = [num2str(fs(i)/1000) ' kHz'];
        cnames{i} = str;
    end

    rnames = {'n','Mean Error dB','Max Error dB','n_{SRL3D}','Mean Error dB','Max Error dB'};
    t = uitable('Parent',f,'Data',T,'ColumnName',cnames,... 
            'RowName',rnames,'Position',[20 20 680 150]);
    set(gcf,'Name',['Based on ' num2str(d(j)) 'm Reference Measurements'])
end

display(' ');
for j = 1:length(d)
    % Display Max difference between measured and predicted calibration
    % parameters.
    display('Max difference between measured and predicted');
    display(['calibration parameters for ' num2str(d(j))  'm measurements:' num2str(max(abs(20.*log10(Ta(1,:,j)./Tb(1,:,j))))) 'dB']);
end

function T = getMeasuredData( fs,d,fc )

for j = 1:length(d)
    for i = 1: length(fs)

            dstr = strrep(num2str(d(j)),'.','_');
            name = ['CalibrationData_' num2str(fs(i)) 'Hz_' dstr 'm.mat'];
            load(name);
        
            % Data is in R
            RIR = R;
    
            % Apply Alias Filter
            b = fir1(100,fc*2);
            Y = filterData(RIR,b);

            E_ref = getEnergy( b );
            E = mean(getEnergy( Y ));

            n(i) = sqrt(E_ref/E);    

            [numIRs tmp] = size(Y);

            fc_i = round(fs(i).*fc);
            if fc_i < 2
              warning(['The bin frequency corresponding to the normalized cut-off frequency ' num2str(fc(i)) ' is either 0 or 1 for sampling frequency ' num2str(fs(i)) ' Hz' ]) 
            end              

            % Source Signal - Reference
            sigr = 20.*log10(abs(fft(b,fs(i))));
            
            % Measured Calibrated Parameter Signal
            sig =  20.*log10(abs(fft(Y(1,:).*n(i),fs(i))));

            % Calc average difference in dB over passband 
            % Measured Calibration parameter
            dif_measured = abs(sigr(1:fc_i)-sig(1:fc_i));
            maxdBerror_measured(i) = max(dif_measured);
            avgdBerror_measured(i) = mean(dif_measured);              
    end
end

T(1,:,j) = n;
T(2,:,j) = avgdBerror_measured;
T(3,:,j) = maxdBerror_measured;

function T = getPredictedData( fs,d,fc,C )

for j = 1:length(d)
    for i = 1: length(fs)

            dstr = strrep(num2str(d(j)),'.','_');
            name = ['CalibrationData_' num2str(fs(i)) 'Hz_' dstr 'm.mat'];
            load(name);
        
            % Data is in R
            RIR = R;
    
            % Apply Alias Filter
            b = fir1(100,fc*2);
            Y = filterData(RIR,b);            

            [numIRs tmp] = size(Y);          

            fc_i = round(fs(i).*fc);
            if fc_i < 2
              warning(['The bin frequency corresponding to the normalized cut-off frequency ' num2str(fc(i)) ' is either 0 or 1 for sampling frequency ' num2str(fs(i)) ' Hz' ]) 
            end              

            % Source Signal
            sigr = 20.*log10(abs(fft(b,fs(i))));
            
            % Predicted Calibrated Parameter Signal
            n_p(i) = fs(i).*C(1) + C(2);
            sigPred =  20.*log10(abs(fft(Y(1,:).*n_p(i),fs(i)))); 
            
            % Calc average difference in dB over passband 
            % Predicted Calibration parameter
            dif_predicted = abs(sigr(2:fc_i)-sigPred(2:fc_i));
            maxdBerror_predicted(i) = max(dif_predicted);
            avgdBerror_predicted(i) = mean(dif_predicted);         
    end
end

T(1,:,j) = n_p;
T(2,:,j) = avgdBerror_predicted;
T(3,:,j) = maxdBerror_predicted;

function getMaxErrorCutOffFreqFigure( fs,d,fc )
% Load the dataset.
for j = 1:length(d)
    for i = 1:length(fs)  
    
        dstr = strrep(num2str(d(j)),'.','_');
        name = ['CalibrationData_' num2str(fs(i)) 'Hz_' dstr 'm.mat'];
        load(name);
        
        % Data is in R
        RIR = R;
        
        for k = 1:length(fc)
        
            % Apply Alias Filter
            b = fir1(100,fc(k)*2);
            Y = filterData(RIR,b);
            
            % Energy of Input Source
            E_ref = getEnergy( b );
            % Energy for each Receiver source
            E = getEnergy( Y );
            % Calibration value as computed using each Receiver signal.
            n = sqrt(E_ref./E);          

            % Source Signal - magnitude response
            sigr = 20.*log10(abs(fft(b,fs(i))));

            % Calc. bin frequency index for the normalized fc. 
            fc_i = round(fs(i).*fc(k));
            if fc_i < 2
                warning(['The bin frequency corresponding to the normalized cut-off frequency ' num2str(fc(i)) ' is either 0 or 1 for sampling frequency ' num2str(fs(i)) ' Hz' ]) 
            end

            % Initialize Arrays
            clear maxdBerror_measured
            maxdBerror_measured(1:length(n)) = 0;

            % For every Receiver...
            for r = 1:length(n)                   
                % Measured Calibrated Parameter Signal - magnitude response
                sig =  20.*log10(abs(fft(Y(r,:).*n(r),fs(i))));

                % Calc average difference in dB over passband 
                % Measured Calibration parameter
                dif_measured = abs(sigr(1:fc_i)-sig(1:fc_i));
                maxdBerror_measured(r) = max(dif_measured);            
            end

            maxmaxdBerror_measured(i,k) = max(maxdBerror_measured);        
          
        end
        
                      
    end
    
    % Create One Figure for every distance 
    % (Only one distance of 1m is shown in article)
    figure;
    s = surf(fc,fs,maxmaxdBerror_measured);
    colormap(1-gray);
    xlabel('Normalized Cut-Off Frequency');
    xlim([0 0.25]);
    ylabel('Sampling Frequency');
    zlabel('Max Error dB');
    title(['d = ' num2str(d(j)) 'm']);
    zlim([0 60]);
end

function E = getEnergy( RIR )
[m n] = size(RIR);
    for i = 1:m
        E(i) = sum(RIR(i,:).^2);
    end

function Y = filterData(RIR,b)
[m n] = size(RIR);
for i = 1:m
    Y(i,:) = filter(b,1,[RIR(i,:) zeros(1,length(n))]);
end