function PARAMS = IEEE_TABLE3_AcousticParams( ACOUSTICPARAMS )

load( ACOUSTICPARAMS );

% Create Table of Acoustic Params
% Reflection Coefficients of Interest
RC = [3 7 9];
models = {'ISM-MEAN','FDTD-MEAN','FDTD2D-MEAN','BTART-MEAN'};

numSrc = 5;
numRec = 5;

% Acoustic Params based on the Single Source and Receiver Location SR
for j = 1:length(RC)
    id = (j-1)*5;
    for i = 1:length(models)
        switch models{i}
            case 'ISM-MEAN'
                
                [numParams numCoeff] = size(ISM{1,1});
                t_accum = zeros(numParams,numCoeff);
                for s = 1:numSrc
                    for r = 1:numRec   
                        for nP = 1:numParams
                            t_accum(nP,:) = t_accum(nP,:) + ISM{s,r}(nP,:);
                        end
                    end
                end                
                D = t_accum(:,RC(j))./(numSrc*numRec); % get mean for every parameter
                
            case 'FDTD-MEAN'  %3d fdtd
                
                [numParams numCoeff] = size(ISM{1,1});
                t_accum = zeros(numParams,numCoeff);
                for s = 1:numSrc
                    for r = 1:numRec   
                        for nP = 1:numParams
                            t_accum(nP,:) = t_accum(nP,:) + FDTD{s,r}(nP,:);
                        end
                    end
                end                
                D = t_accum(:,RC(j))./(numSrc*numRec); % get mean for every parameter
                
            case 'FDTD2D-MEAN'
                
                [numParams numCoeff] = size(ISM{1,1});
                t_accum = zeros(numParams,numCoeff);
                for s = 1:numSrc
                    for r = 1:numRec   
                        for nP = 1:numParams
                            t_accum(nP,:) = t_accum(nP,:) + FDTD2D{s,r}(nP,:);
                        end
                    end
                end                
                D = t_accum(:,RC(j))./(numSrc*numRec); % get mean for every parameter
                
            case 'BTART-MEAN'  %BT ART
                
                [numParams numCoeff] = size(ISM{1,1});
                t_accum = zeros(numParams,numCoeff);
                for s = 1:numSrc
                    for r = 1:numRec   
                        for nP = 1:numParams                            
                            if (s == 3 && sum(r == [2 3 4 5]) == 1)
                                %omit 12 13 14 15 -  REMEMBER TO REMOVE -4
                                %BELOW WHEN THIS IS FIXED.
                            else
                                t_accum(nP,:) = t_accum(nP,:) + BTART{s,r}(nP,:);
                            end
                        end
                    end
                end                
                D = t_accum(:,RC(j))./ ((numSrc*numRec)-4) ; % get mean for every parameter
                
            case 'FDTD'  %3d fdtd
                D = FDTD{SR(1),SR(2)}(:,RC(j));
            case 'FDTD2D'
                D = FDTD2D{SR(1),SR(2)}(:,RC(j));
            case 'ISM'
                D = ISM{SR(1),SR(2)}(:,RC(j));           
            case 'BT-ART'
                D = BTART{SR(1),SR(2)}(:,RC(j));
            otherwise
               error(['The specified acoustic parameter dataset ' models{i} ' does not exist.']);
        end

        T(id+1,i) = D(5)*1000; %T30
        T(id+2,i) = D(4)*1000; %EDT
        T(id+3,i) = D(3)*1000; %CT
        T(id+4,i) = D(2); %C80
        T(id+5,i) = D(1); %C50    
        
    end
    
    % Row Names
    rows(id+1:id+5) = {'T30','EDT','CT','C80','C50'};
       
    % Add JND Column 10%
    jcol = length(models)+1;
    T(id+1,jcol) = mean( [T(id+1,1) T(id+1,3)] )*0.1; % 10percent of Mean T30 of FDTD and ISM
    T(id+2,jcol) = mean( [T(id+2,1) T(id+2,3)] )*0.05; % 5percent of Mean EDT of FDTD and ISM
    T(id+3,jcol) = 10;
    T(id+4,jcol) = 1;
    T(id+5,jcol) = 1;     
    
end

PARAMS = T;

% Column Names in Article
models = {'ISM','3DFD','2DFD','BT-ART'};
cols = models;
cols{length(models)+1} = 'JND';

% Generate Reflection Coefficients
R = -1:0.1:1;
R(1) = R(1)+0.001;
R(end) = R(end)-0.001;
R = R(11:21);

table2latex(PARAMS,R(RC),rows,cols);

function table2latex(PARAMS,RC,rows,cols)

[nrows,ncols]= size(PARAMS);

filename = 'AcoustciParamsTable3.tex';
display(['Writing Table 3 Acoustic Parameters to a LaTeX file ' filename]);
display(['located in folder ' cd]);

fid = fopen(filename, 'w');

fprintf(fid,'\\begin{table}[ht]\n');
fprintf(fid,'\\caption{Other params}\n');
fprintf(fid,'\\centering\n');
fprintf(fid,'\\small\n');
fprintf(fid,'\\setlength{\\tabcolsep}{0.1cm}\n');
fprintf(fid,'\\label{paramsTable}\n');
fprintf(fid,'\\begin{tabular}{cc}\n');
fprintf(fid,'\\begin{tabular}{c}\n');
fprintf(fid,'\\\\ \\\\ (a) $%1.1f$',RC(1));
letstr = {'(a)','(b)','(c)','(d)','(e)','(f)','(g)','(h)','(i)'};
for i = 2:length(RC)
    str = ['\\\\ \\\\ \\\\ \\\\ \\\\ ' letstr{i} ' $%1.1f$'];
    fprintf(fid,str,RC(i));
end
fprintf(fid,'\n');

fprintf(fid,'\\end{tabular}&\n');
fprintf(fid,'\\begin{tabular}{c|');
for i = 1:ncols
    fprintf(fid,'c');
end
fprintf(fid,'}\n');

fprintf(fid,'Param. ');
for i = 1:length(cols)
    fprintf(fid,[' & ' cols{i} ]);
end
fprintf(fid,'\\\\\\hline\\hline\n');

for t = 1:length(RC)
    rid = (t-1)*(nrows/length(RC));
    for i = 1:(nrows/length(RC))        
        fprintf(fid,[rows{i}]);
        for j = 1:ncols
            switch cols{j}
                case 'JND'
                    if i == 1
                        % Add 30 percent value to T30 row
                        fprintf(fid,'& %3.0f/%3.0f',PARAMS(rid+i,j),PARAMS(rid+i,j)*3);
                    else
                        fprintf(fid,'& %3.1f',PARAMS(rid+i,j));
                    end
                otherwise
                    fprintf(fid,'& %3.1f',PARAMS(rid+i,j));
            end
        end  
        fprintf(fid,'\\\\ \n');
    end
    fprintf(fid,'\\hline\n');
end

fprintf(fid,'\\end{tabular}\n');
fprintf(fid,'\\end{tabular}\n');
fprintf(fid,'\\end{table}\n');

fclose(fid);