function alpha = getAirAbsorption(ff)
% Function to correcly return the absorption of sound in still air at 20C 
% and is valid in the frequency range 12Hz - 1MHz over all humidities as 
% given by Bass et al. 
% JASA Feb 1972 - Atmospheric Absorption of Sound: Analytical Expressions
%
% NOTE: The authors have made more recent papers which are not taken to
% account in this script.
%
%   Usage:
%   ff =    The frequency of interest in Hz- this may be 
%           a vector of frequencies.
%   alpha = absorption is given as dB of attenuation for 
%           each ff over 304m (1000ft).
%
% Implemented by, 
% A. Southern, 
% Virtual Acoustics Team
% Dept of Media Technology, 
% Aalto University, March 2012

% Note: Denoising Code
% http://homepage.univie.ac.at/moika.doerfler/StrucAudio.html

if nargin == 0
    ff = 20:20000;
end

H = 45; % Percentage Humidity
A = get_A(H);
T = get_T(H);
c = get_c();

for k = 1:length(ff)
    
    f = ff(k);
    
    for i = 1:4
        X = f*T(i)*A(i);
        Y = 1 + (f^2) * (T(i)^2);
        Z = X/Y;
        W(i) = (Z+(1.525E-9)*f);
    end

    alpha(k) = 27.26*sum(W)*(f/c);

end

function plot_AandT
% The function produces same plots as in the paper
H = 0.01:0.01:100;
A(1:length(H),1:4) = 0;
T(1:length(H),1:4) = 0;

for n = 1:length(H)
    A(n,:) = get_A(H(n));
    T(n,:) = get_T(H(n));
end

h1 = figure;
h2 = figure;
for n = 1:3
    figure(h1);
    subplot(3,1,n);
    plot(H,A(:,n));
    if n == 1 
        title('Derived Relaxtion Strengths');
    end

    figure(h2);
    hold on;
    plot(H,T(:,n));
    if n == 1
        title('Modified Relaxtion Times');
    end 
end

figure(h2);
set(gca,'Yscale','log');
ylim([1E-7 2E-1]);

function A = get_A(H)
A_b = [-8.97433500000000,-0.00320434600000000,-0.000472068800000000,-0.000152587900000000;-7.39732400000000,0.00617981000000000,0.000112533600000000,-1.04904200000000e-05;-10.4035500000000,0.0169830300000000,-0.00246810900000000,-0.000279426600000000;];

for i = 1:3
    b0 = A_b(i,1);
    b1 = A_b(i,2);
    b2 = A_b(i,3);
    b3 = A_b(i,4);    
    A(i) = exp( b0 + b1*log(H) + b2*(log(H)^2) + b3*(log(H)^3));
end
A(4) = 1;

function T = get_T(H)
T_b = [-2.35700900000000,-0.542330700000000,-0.0500000000000000,5E-10;-5.38899200000000,-1.23114000000000,-0.0476942100000000,0.00400006800000000;-9.78059400000000,-0.845981000000000,-0.0339984900000000,0.00253295900000000;];

for i = 1:3
    b0 = T_b(i,1);
    b1 = T_b(i,2);
    b2 = T_b(i,3);
    b3 = T_b(i,4);
    T(i) = exp( b0 + b1*log(H) + b2*(log(H)^2) + b3*(log(H)^3) );
end
T(4) = 5E-10;

function c = get_c(Pw,H,theta)

if nargin == 0
    c = 1.13;
else
    h = Pw*H/100;
    F = 28.966-10.95*h;
    E = 8.3166E7 * theta;
    D = (E / F);
    C = 2.5 + 5*h;
    B = 3.5 + 5*h;
    A = ( (B/C)*D ); 
    c = (1/30480)*( A^0.5 );
end