 function y = applyAirAbsorption(x,fs,fc,winspan,nTaps)
% function assumes that the time sample is explicitly gives the distance
% travelled.
% Air absorption is applied in time/frequency domain as a real-time
% convolutions
%
% fc is optional, air absorption is only applied above fc.
% e.g. y = applyAirAbsorption(x,fs,[]);
%      y = applyAirAbsorption(x,fs,1000);
%
% By Alex Southern
% Virtual Acoustic Team
% Department of Media Technology
% School of Science
% Aalto University
% Espoo
% Finland

% Define Frame size (and window length)
if exist('winspan') == 0
    winspan = 1024;
end

if exist('nTaps') == 0
    nTaps = 100;
end

%Get the filter length
h = fir1(nTaps,0.99,'low');   % all pass - low pass filter
L = length(h);              % FIR filter length in taps

% get input signal length and pad beginning and end
origsignal_length = length(x);
x = [zeros(winspan+L-1,1); x; zeros(winspan+L-1,1)];
signal_length = length(x);

% Create Window
wintype = 0;
switch wintype
    case 0          % rectangular
        winshift = winspan;      % hop size
        win = ones(winspan,1);
    case 1          % hanning (50% overlap)
        winshift = winspan/2;    % hop size
        win = hanning(winspan,'periodic');
end

%FFT processing parameters: 
fftsize = 2^(ceil(log2(winspan+L-1)));          % FFT Length        

numFrames = 1+floor((signal_length-winspan)/winshift);    % no. complete frames
y = zeros(1,signal_length + fftsize);     % allocate output+'ringing' vector
specsize = fftsize/2 +1;

% FFT lowpass filter
hzp = [h'; zeros(1,fftsize-L)'];          % zero-pad h to FFT size
H = fft(hzp);                             % filter frequency response
H = H(1:specsize);

% Problem Specific Initialization
ff = ((1:specsize).*fs)/fftsize; % Air attenuation band in Hz
%--------------------------------

if isempty(fc) == 0
    % fc = absorption is only applied above this frequency (Hz)
    fci = round((fc*fftsize)/fs); %bin frequency of fc    
end

i = 1;
framecount = 1;
while i + fftsize <= signal_length + fftsize && framecount <= numFrames
        
    % Apply window in time domain and the take the FFT.
    pad_sig = [win.*x(i:(i+winspan-1)); zeros(fftsize-winspan,1)];
    SIG = fft( pad_sig );  
    SIG = SIG(1:specsize,:); % Only need the first half           
    
    % Main Processing ---------------

    alpha = airAttenuationCoefficient(i+(winshift/2),fs,ff)';
    if isempty(fc) == 0
        alpha(1:fci) = 1;
    end
    
    Z = SIG.*H.*alpha; 
   
    % -------------------------------
    
    % Convert Magnitude and Phase back to Complex Signal
    %Z = mag_n.*exp(1i*ph);
    
    % Create Second Half
    tmp = complex(real(Z(2:end-1)),-imag(Z(2:end-1)));
      
    %Convert window back to Time Domain
    y(i:(i+fftsize-1)) =   real(ifft(cat(1,Z,flipud(tmp))))' + y(i:(i+fftsize-1));
    
    % Increment counters
    i = i + winshift;
    framecount = framecount+1;
end

% remove pad from beginning
y = y(winspan+L:end);
% remove filter delay
y = y(51:end);
% match input length
y = y(1:origsignal_length-1);


function alpha = airAttenuationCoefficient(timeframe,fs,ff);
d = 344*(timeframe/fs); % distance travelled
alpha = NormalizedAirAbsorption( d, ff );

